<?php

declare(strict_types=1);

namespace NeuronAI\RAG\VectorStore;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;
use NeuronAI\RAG\Document;
use NeuronAI\RAG\VectorSimilarity;

use function count;
use function in_array;
use function is_null;
use function json_decode;
use function trim;
use function uniqid;

class ChromaVectorStore implements VectorStoreInterface
{
    protected Client $client;

    protected string $collectionId;

    /**
     * @throws GuzzleException
     */
    public function __construct(
        protected string $collection,
        protected string $host = 'http://localhost:8000',
        protected string $tenant = 'default_tenant',
        protected string $database = 'default_database',
        protected ?string $key = null,
        protected int $topK = 5,
    ) {
        $this->initialize();
    }

    /**
     * Create the collection if it doesn't exist
     *
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function initialize(): void
    {
        $response = $this->client()->post(trim($this->host, '/')."/api/v2/tenants/{$this->tenant}/databases/{$this->database}/collections", [
            RequestOptions::JSON => [
                'name' => $this->collection,
                'get_or_create' => true,
            ]
        ])->getBody()->getContents();

        $this->collectionId = json_decode($response, true)['id'];
    }

    protected function client(): Client
    {
        return $this->client ?? $this->client = new Client([
            'base_uri' => trim($this->host, '/')."/api/v2/tenants/{$this->tenant}/databases/{$this->database}/collections/",
            'headers' => [
                'Content-Type' => 'application/json',
                ...(!is_null($this->key) && $this->key !== '' ? ['Authentication' => 'Bearer '.$this->key] : [])
            ]
        ]);
    }

    /**
     * @throws GuzzleException
     */
    public function addDocument(Document $document): VectorStoreInterface
    {
        return $this->addDocuments([$document]);
    }

    /**
     * @throws GuzzleException
     */
    public function deleteBySource(string $sourceType, string $sourceName): VectorStoreInterface
    {
        $this->client()->post("{$this->collectionId}/delete", [
            RequestOptions::JSON => [
                'where' => [
                    '$and' => [
                        ['sourceType' => ['$eq' => $sourceType]],
                        ['sourceName' => ['$eq' => $sourceName]]
                    ]
                ]
            ]
        ]);

        return $this;
    }

    /**
     * Delete the current collection
     *
     * @throws GuzzleException
     */
    public function destroy(): void
    {
        $this->client()->delete($this->collection);
    }

    /**
     * @throws GuzzleException
     */
    public function addDocuments(array $documents): VectorStoreInterface
    {
        $this->client()->post("{$this->collectionId}/add", [
            RequestOptions::JSON => $this->mapDocuments($documents),
        ]);

        return $this;
    }

    /**
     * @throws GuzzleException
     */
    public function similaritySearch(array $embedding): iterable
    {
        $response = $this->client()->post("{$this->collectionId}/query", [
            RequestOptions::JSON => [
                'query_embeddings' => [$embedding],
                'n_results' => $this->topK,
                'include' => ['documents', 'metadatas', 'distances'],
            ]
        ])->getBody()->getContents();

        $response = json_decode($response, true);

        // Map the result
        $size = count($response['ids'][0] ?? []);
        $result = [];
        for ($i = 0; $i < $size; $i++) {
            $document = new Document();
            $document->id = $response['ids'][0][$i] ?? uniqid();
            //$document->embedding = $response['embeddings'][0][$i] ?? null;
            $document->content = $response['documents'][0][$i];
            $document->sourceType = $response['metadatas'][0][$i]['sourceType'] ?? null;
            $document->sourceName = $response['metadatas'][0][$i]['sourceName'] ?? null;
            $document->score = VectorSimilarity::similarityFromDistance($response['distances'][0][$i] ?? 0.0);

            foreach (($response['metadatas'][0][$i] ?? []) as $name => $value) {
                if (!in_array($name, ['content', 'sourceType', 'sourceName', 'score', 'embedding', 'id'])) {
                    $document->addMetadata($name, $value);
                }
            }

            $result[] = $document;
        }

        return $result;
    }

    /**
     * @param Document[] $documents
     */
    protected function mapDocuments(array $documents): array
    {
        $payload = [
            'ids' => [],
            'documents' => [],
            'embeddings' => [],
            'metadatas' => [],
        ];

        foreach ($documents as $document) {
            $payload['ids'][] = (string) $document->getId();
            $payload['documents'][] = $document->getContent();
            $payload['embeddings'][] = $document->getEmbedding();
            $payload['metadatas'][] = [
                'sourceType' => $document->getSourceType(),
                'sourceName' => $document->getSourceName(),
                ...$document->metadata,
            ];
        }

        return $payload;
    }
}
